
(function() {

  const ns = laaBookmarksTitleStyles;   /* namespace */

  ns.SmartCompareExt = function(name1,name2) {
    return ns.SmartCompare(name1,name2,true);
  }

  ns.SmartCompare = function(name1,name2,Ext) {
    if (!Ext) { Ext = false; }
    
    //*********************************************************************************************************
    //
    // SmartCompare Function
    //
    // Compares two file names or paths to determine Smart sort order, as detailed below.
    //
    // See discussion below for use of the optional "Ext" argument.
    //
    // If name1 > name2 then function returns 1.
    // If name1 = name2 then function returns 0.
    // If name1 < name2 then function returns -1.
    //
    // ********************
    //
    // Text sequence is the order in which the file names would be retrieved from disk
    // if using js folder processing methods - different from the Windows Browse Smart order.
    //
    // For Text sequence, the following principles apply:
    //
    // - A shorter name that is a leading substring of another name sorts first -
    //   e.g., xxx, xxx1, xxxx.
    //
    // - The characters in the same positions in the two names are sorted in their binary ANSI
    //   code order, except that similar uppercase and lowercase letters are treated as equal.
    //
    // - The names are scanned left to right.  The first character difference determines the
    //   sort order.
    //
    // ********************
    //
    // Smart order is the sequence in which Windows displays files during Browse.  The most
    // significant part of Smart order is the sequencing of file names that contain numeric
    // strings in corresponding positions, even when the numeric strings vary in length.
    // This routine closely approximates Smart order, by applying observed Windows XP criteria.
    //
    // Smart order applies several special criteria, and then otherwise sorts the names in
    // Text sequence.
    //
    // Smart order differs from simple Text sequence in the following ways:
    //
    // - Certain special characters sort low, before any other characters, including numbers
    //   and alpha, when they are in the first character position of the file or folder name.
    //
    // - Numeric digits sort before special characters and alpha, except as noted above.
    //   Examples:  X1, X2, X.A, X-A, XA, XB.
    //
    // - Certain special characters sort in a special order relative to each other, different
    //   from their ANSI code order.
    //
    // - Multi-digit numeric strings are recognized when they start in the same position
    //   in names, and are sorted numerically - e.g., X1, X2, X10, X20.
    //
    // - Two names may be numerically identical by the above criteria, but physically different
    //   due to different numbers of leading zeros in corresponding numeric string positions.
    //   In this case, an equal-value number with more leading zeros gets sorted first -
    //   e.g., X001, X01, X1.  However, this criteria is only applied after the entire names
    //   have been fully scanned for other differences.  Two logically identical names are then
    //   sorted via a simple Text comparison of the two names.  This puts leading zeros in the
    //   indicated order.
    //
    // - In Windows Browse, Smart order may be affected by whether the user has set the option
    //   to display file name extensions.  For example:
    //
    //       With extensions hidden (default):  MemoX, MemoX1, MemoXA
    //       With extensions displayed:         MemoX1.docx, MemoX.docx, MemoXA.docx
    //
    //   The above difference is caused by the convention that shorter names, like XXX, are
    //   supposed to sort before longer names of which they are a leading substring, such as
    //   XXX1.  But with extensions displayed, none of the names are substrings of the others.
    //
    //   The optional third argument, "Ext", passed to this function, may be true or false,
    //   and determines how file name extensions are processed:
    //
    //   a) If Ext is omitted or false, this function does not attempt to recognize or specially
    //      process extensions such as "docx".  The result is that the names will be sorted as
    //      in the two examples above, according to whether the extensions are included in the
    //      file names passed to the function.
    //
    //   b) If Ext is true, then this function assumes that file name extensions are present
    //      in the two names passed.  Shorter names such as MemoX.docx that logically are
    //      leading substrings of names such as MemoX1.docx will be sorted first.
    //
    // ********************
    //
    // This java script module was built by converting a similar VBA module in the most straight-forward manner.
    // As a result, some of the coding style in this module is slightly unusual for java script, and it likely
    // could be made a bit more compact with slightly improved performance. But it should get the desired result.
    //
    // Keeping it similar to the other language versions makes for easier parallel maintenance, if ever needed.
    //
    // Processing of forward slashes ("/") in paths was added to this version for cross-platform support.
    //
    //*********************************************************************************************************

    const numeric = /^[0-9]+$/;   // Returns true if string consists of one or more numerals
    
    var x1, x2, c1, c2, islead;
      
    name1 = name1.toLowerCase();   // Treat uppercase and lowercase alpha as equal for sorting
    name2 = name2.toLowerCase();

    x1 = -1;
    x2 = -1;
    
    //***********************************************

    while (true) {   // Loop continues until exited with a break or return statement

      x1 = x1 + 1;
      x2 = x2 + 1;

      c1 = name1.substr(x1,1);
      c2 = name2.substr(x2,1);

      // Check for the ends of file and folder names

      if ( (c1 == "\\") && (c2 == "\\") ) { continue; }   // End of folder names (\), continue
      if (c1 == "\\") { return -1; }    // Folder name in name1 is shorter
      if (c2 == "\\") { return 1; }   // Folder name in name1 is longer

      if ( (c1 == "/") && (c2 == "/") ) { continue; }   // End of folder names (/), continue
      if (c1 == "/") { return -1; }    // Folder name in name1 is shorter
      if (c2 == "/") { return 1; }   // Folder name in name1 is longer

      if ( (c1 == "") && (c2 == "") ) { break; }   // Two logically identical names.
                                                    // May differ due to leading zeros in nums.
      if (c1 == "") { return -1; }   // name1 is shorter
      if (c2 == "") { return 1; }    // name1 is longer

      if (Ext == true) {   // If Ext argument is true, then process extensions
      
        if ( (name1.indexOf("\\",x1) == -1) && (name2.indexOf("\\",x2) == -1) &&
             (name1.indexOf("/",x1) == -1)  && (name2.indexOf("/",x2) == -1) ) {
            // Past folder names, and scanning two file names

          if ( (name1.lastIndexOf(".") == x1) && (name2.lastIndexOf(".") == x2) ) { continue; }
              // Arrived at both extensions, so continue sorting

          if (name1.lastIndexOf(".") == x1) { return -1; }   // name1 is shorter
          if (name2.lastIndexOf(".") == x2) { return 1; }    // name1 is longer
        }
      }

      // Check special leading characters of the two file or folder names

      if (x1 == 0) { islead = true; }
      
      else {
        islead = false;
        if (name1.substr(x1-1,1) == "\\") { islead = true; }   // Preceding char is backslash
        if (name1.substr(x1-1,1) == "/") { islead = true; }    // Preceding char is forward slash
      }
      
      if ( (islead) && (c1 != c2) ) {

        if (c1 == "#") { return -1; }
        if (c2 == "#") { return 1; }

        if (c1 == "@") { return -1; }
        if (c2 == "@") { return 1; }

        if (c1 == "+") { return -1; }
        if (c2 == "+") { return 1; }

        if (c1 == "=") { return -1; }
        if (c2 == "=") { return 1; }
      }

      // Check for numeric characters

      if ( (numeric.test(c1)) && (numeric.test(c2)) ) {   // Have 1st digit of numeric strings in both names

        while (numeric.test(name1.substr(x1+1,1))) {   // Get all digits
          x1 = x1 + 1;
          c1 = c1 + name1.substr(x1,1);
        }

        while (numeric.test(name2.substr(x2+1,1))) {   // Get all digits
          x2 = x2 + 1;
          c2 = c2 + name2.substr(x2,1);
        }

        if (Number(c1) < Number(c2)) { return -1; }   // name1 has lower number
        if (Number(c1) > Number(c2)) { return 1; }    // name1 has higher number
        continue;   // Numeric strings are equal in value, keep scanning
      }

      if (numeric.test(c1)) { return -1; }   // Sort numbers before other characters
      if (numeric.test(c2)) { return 1; }    // Sort numbers before other characters

      // Check for certain special characters

      if (c1 != c2) {

        if (c1 == " ") { return -1; }
        if (c2 == " ") { return 1; }

        if (c1 == "#") { return -1; }
        if (c2 == "#") { return 1; }

        if (c1 == "-") { return -1; }
        if (c2 == "-") { return 1; }

        if (c1 == ".") { return -1; }
        if (c2 == ".") { return 1; }

        if (c1 == "@") { return -1; }
        if (c2 == "@") { return 1; }

        if (c1 == "+") { return -1; }
        if (c2 == "+") { return 1; }
      }

      if (c1 < c2) { return -1; }   // name1 character sorts low
      if (c1 > c2) { return 1; }    // name1 character sorts high
      
      // Names equal so far, so loop and keep scanning
      
    }   // End of main while loop
    
    //***********************************************

    // Do text comparison now that no logical inequalities have been found.
    // Names that are numerically equal, but physically differ, can still sort high or low, due to leading zeros.

    if (name1 < name2) { return -1; }   // name1 sorts low
    if (name1 > name2) { return 1; }    // name1 sorts high
    return 0;                           // names are equal
  }

})();
