'use strict';

const tabsList = document.getElementById( 'tabs-list' );
const tabsHistory = document.getElementById( 'tabs-history' );
const createTab = document.getElementById( 'create-tab' );
const openOptions = document.getElementById( 'options' );
const hideAllTabs = document.getElementById( 'hide-all-tabs' );

const options = { 
	localeMatcher: 'best fit',
	weekday: 'short', 
	year: 'numeric', 
	month: '2-digit', 
	day: 'numeric', 
	hour: '2-digit', 
	minute: '2-digit'
};

const locale = window.navigator.languages[0];

/*******************************************************************************/
function getCurrentWindowTabs() {
	return browser.tabs.query( { currentWindow: true } );
}

function listTabs() {
	getCurrentWindowTabs().then( ( tabs ) => {
		let currentTabs = document.createDocumentFragment();
		tabsList.textContent = '';
		for(let tab of tabs) {
			let tabLink = document.createElement( 'a' );
			tabLink.textContent = tab.title || tab.id;
			tabLink.setAttribute( 'href' , tab.id );
			tabLink.classList.add( 'switch-tabs' );

			let favIconURL = tab.favIconUrl;
			if( favIconURL && favIconURL != '' ) {
				tabLink.style.backgroundImage = 'url("' + favIconURL + '")';
			} else {
				tabLink.classList.add( 'default' );
			}
			if( tab.active ) {
				tabLink.classList.add( 'active' );
			}
			if( tab.discarded ) {
				tabLink.classList.add( 'discarded' );
			}	
			if( tab.hidden ) {
				tabLink.classList.add( 'hidden' );
			}				
			currentTabs.appendChild( tabLink );
		}
		tabsList.appendChild( currentTabs );
	}).then( () => { sessionList(); } );
}

/*******************************************************************************/
document.addEventListener( 'click' , (e) => {
	if( e.target.classList.contains( 'switch-tabs' ) ) {
		var tabId = +e.target.getAttribute( 'href' );
		browser.tabs.query({
			currentWindow: true
		}).then( ( tabs ) => {
			for( var tab of tabs ) {
				if( tab.id === tabId ) {
					if(e.button === 0) {
						browser.tabs.update( tabId , {
							active: true
						});
					}
					if(e.button === 1) {
						browser.tabs.remove( tabId );
					}
					if(e.button === 2) {
						/* Hidden tabs are not automatically discarded: the code 
						running in them continues to run. However, it's 
						recommended that you should also discard hidden tabs, to 
						help manage memory and resource usage.
						https://developer.mozilla.org/en-US/Add-ons/WebExtensions/API/tabs/hide
						https://developer.mozilla.org/en-US/Add-ons/WebExtensions/API/tabs/discard
						*/
						browser.tabs.discard( tabId );
						/*
						It is only supported in Firefox and is disabled by default. 
						To enable it, visit about:config and set 
						"extensions.webextensions.tabhide.enabled" 
						to true.
						*/
						browser.tabs.hide( tabId );
					}
				}
			}
		});//.then( () => { listTabs(); } );
	}
	if( e.target.classList.contains( 'switch-history' ) ) {
		var historyURL = e.target.getAttribute( 'href' );
		browser.tabs.create({
			url: historyURL
		});
	}
	e.preventDefault();
});

/*******************************************************************************/

function updateList() {
	//setTimeout( listTabs , 250 );
	listTabs();
}

/*******************************************************************************/
createTab.addEventListener( 'click' , (e) => {
	browser.tabs.create({});
});

openOptions.addEventListener( 'click' , (e) => {
	browser.tabs.create({
		url: '../options/options.html'
	});
});

hideAllTabs.addEventListener( 'click' , (e) => {
	browser.tabs.query({
		currentWindow: true
	}).then( ( tabs ) => {
		for( var tab of tabs ) {
			browser.tabs.discard( tab.id );
			browser.tabs.hide( tab.id );
		}
	});//.then( () => { listTabs(); } );
});

document.addEventListener( 'contextmenu' , (e) => {
	e.preventDefault();
});

document.addEventListener( 'DOMContentLoaded' , listTabs );

browser.tabs.onRemoved.addListener( updateList );
browser.tabs.onCreated.addListener( updateList );
browser.tabs.onActivated.addListener( updateList );
browser.tabs.onMoved.addListener( updateList );
browser.tabs.onUpdated.addListener( updateList );

/*******************************************************************************/

function sessionList() {
	var gettingSessions = browser.sessions.getRecentlyClosed({
		maxResults: 10
	});
	gettingSessions.then( listMostRecent );
}

sessionList();

function listMostRecent( sessionInfos ) {
	if( !sessionInfos.length ) {
		return;
	}
	tabsHistory.textContent = '';
	let currentHistory = document.createDocumentFragment();
	for( let sessionInfo of sessionInfos ) {
		
		let history = document.createElement( 'a' );

		var timestr = new Date( sessionInfo.tab.lastAccessed ).toLocaleDateString( locale , options );
		
		history.innerHTML = (sessionInfo.tab.title || browser.i18n.getMessage('Untitled')) + '<br /> (' + browser.i18n.getMessage('LastVisit') + ': ' + timestr + ')';
		history.setAttribute( 'href' , sessionInfo.tab.url );
		
		let favIconURL = sessionInfo.tab.favIconUrl;
				if( favIconURL && favIconURL != '' ) {
					history.style.backgroundImage = 'url("' + favIconURL + '")';
				} else {
					history.style.backgroundImage = 'url("../icons/icon-default.png")';
				}
		
		history.classList.add( 'switch-history' );
		currentHistory.appendChild( history );
		
	}
	tabsHistory.appendChild( currentHistory );//.then( () => { listTabs(); } );
}

/*
let sessionInfo = sessionInfos[0];
console.log( sessionInfo );
console.log( sessionInfo.lastModified );
console.log( sessionInfo.tab.title );
console.log( sessionInfo.tab.favIconUrl );
console.log( sessionInfo.tab.lastAccessed );
console.log( sessionInfo.tab.url );
console.log( sessionInfo.tab.sessionId );
*/

/*
function onGot( historyItems ) {
	tabsHistory.textContent = '';
	let currentHistory = document.createDocumentFragment();
	for ( item of historyItems ) {
		
		let history = document.createElement( 'a' );

		var timestr = new Date( item.lastVisitTime ).toLocaleDateString( locale , options );
		
		history.innerHTML = (item.title || 'Untitled') + '<br /> (' + browser.i18n.getMessage('LastVisit') + ': ' + timestr + ')';
		history.setAttribute( 'href' , item.url );
		history.classList.add( 'switch-history' );
		currentHistory.appendChild( history );
		
	}
	tabsHistory.appendChild( currentHistory );
	//listTabs();
}

function searchHistory() {
	var searching = browser.history.search( { text: "http" , maxResults: 5 } );
	searching.then( onGot );
}
*/




